'''
Implementacja wersji aplikacji Modny ciuch w Python Command Shell
'''

from ShellUI.BTCInput import *

from  Storage.StockItem import StockItem

class FashionShopShell:
    '''
    Provides a Command Shell interface for use by
    the Fashion Shop application
    '''

    def __init__(self, filename, storage_class):
        '''
        Zarządza danymi sklepu z odzieżą
        Wyświetla komunikat, jeśli ładowanie się nie powiedzie i tworzy nowy sklep
        '''
        FashionShopShell.__filename = filename
        try:
            self.__shop = storage_class.load(filename)
        except:
            print('Nie załadowano danych sklepu z odzieżą.')
            print('Tworzenie pustego sklepu z odzieżą')
            self.__shop = storage_class()

    def create_new_stock_item(self):
        '''
        Tworzy nowy towar magazynowy Pobiera szczegóły towaru, 
        tworzy go, a następnie zapisuje w sklepie
        '''

        stock_ref = read_text('Wprowadź numer magazynowy: ')
        price = read_float_ranged(prompt='Wprowadź cenę: ',
                                    min_value=StockItem.min_price,
                                    max_value=StockItem.max_price)
        tag_string = read_text('Wprowadź znaczniki (rozdzielone przecinkami): ')

        tags = StockItem.get_tag_set_from_text(tag_string)

        new_item = StockItem(stock_ref=stock_ref, price=price, tags=tags)

        try:
            self.__shop.store_new_stock_item(new_item)
            print('Towar zapisano')
        except Exception as e:
            print('Nie zapisano towaru')
            print(e)

    def add_stock(self):
        '''
        Dodaje towar do istniejącej pozycji magazynowej
        Najpierw wyszukuje towar, a następnie pobiera 
        liczbę pozycji towaru do dodania
        '''
        print('Dodaj towar')

        item_stock_ref = read_text('Wprowadź numer magazynowy: ')

        item = self.__shop.find_stock_item(item_stock_ref)
        
        if item == None:
            print('Nie znaleziono towaru')
            return

        print(item)

        number_to_add = read_int_ranged('Ilość towaru do dodania (0 aby zrezygnować): ',
                                        0, StockItem.max_stock_add)

        if number_to_add == 0:
            print('Nie dodano towaru')
        else:
            item.add_stock(number_to_add)
            print(item)

    def sell_stock(self):
        '''
        Sprzedaż towaru Najpierw wyszukuje towar, a następnie odczytuje
        liczbę sztuk towaru do sprzedaży.
        Nie pozwala na sprzedaż większej ilości towarów niż ma na stanie
        '''
        print('Sprzedaż towaru')

        item_stock_ref = read_text('Wprowadź numer magazynowy: ')

        item = self.__shop.find_stock_item(item_stock_ref)

        if item == None:
            print('Nie znaleziono towaru')
            return

        print('Sprzedaż')
        print(item)

        if item.stock_level == 0:
            print('Nie ma towarów w magazynie')
            return

        number_sold = read_int_ranged('Ilość towaru do sprzedaży (0 aby zrezygnować): ',
                                      0,
                                      item.stock_level)

        if number_sold == 0:
            print('Zrezygnowano ze sprzedaży')
            return

        item.sell_stock(number_sold)

        print('Sprzedano towar')

    def do_report(self):
        print('Raport magazynowy')
        print(self.__shop)

    def do_tag_filter(self):
        print('Filtrowanie wg znaczników')
        tag_string = read_text('Wprowadź znaczniki (rozdzielone przecinkami): ')
        search_tags = StockItem.get_tag_set_from_text(tag_string)
        items = self.__shop.find_matching_with_tags(search_tags)
        stock = map(str,items)
        stock_list = '\n'.join(stock)
        template = '''Pasujące towary

{0}
'''
        print(stock_list)

    def main_menu(self):

        prompt = '''Sklep Modny Ciuch

1: Utwórz nowy towar magazynowy
2: Dodaj zapas do istniejącej pozycji
3: Sprzedaj towar
4: Raport ze sprzedaży
5: Wyszukaj wg znaczników
6: Wyjście

Wprowadź polecenie: '''

        while(True):
            command = read_int_ranged(prompt, 1, 7)
            if command == 1:
                self.create_new_stock_item()
            elif command == 2:
                self.add_stock()
            elif command == 3:
                self.sell_stock()
            elif command == 4:
                self.do_report()
            elif command == 5:
                self.do_tag_filter()
            elif command == 6:
                self.__shop.save(FashionShopShell.__filename)
                print('Zapisano dane sklepu z odzieżą')
                break
